#!/bin/bash

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"
OUTPUT_DIR="${PROJECT_DIR}/dist-package"
PACKAGE_NAME="voiceforge-deploy"

log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

echo -e "${BLUE}"
echo "╔═══════════════════════════════════════════════════════════╗"
echo "║          VoiceForge Package Builder                       ║"
echo "╚═══════════════════════════════════════════════════════════╝"
echo -e "${NC}"

cd "$PROJECT_DIR"

log_info "Cleaning previous package..."
rm -rf "$OUTPUT_DIR"
mkdir -p "$OUTPUT_DIR/voiceforge"

log_info "Copying source files..."
cp -r client "$OUTPUT_DIR/voiceforge/"
cp -r server "$OUTPUT_DIR/voiceforge/"
cp -r shared "$OUTPUT_DIR/voiceforge/"
cp -r deploy "$OUTPUT_DIR/voiceforge/"

cp package.json "$OUTPUT_DIR/voiceforge/"
cp package-lock.json "$OUTPUT_DIR/voiceforge/" 2>/dev/null || true
cp tsconfig.json "$OUTPUT_DIR/voiceforge/"
cp vite.config.ts "$OUTPUT_DIR/voiceforge/"
cp tailwind.config.ts "$OUTPUT_DIR/voiceforge/"
cp postcss.config.js "$OUTPUT_DIR/voiceforge/"
cp drizzle.config.ts "$OUTPUT_DIR/voiceforge/"
cp theme.json "$OUTPUT_DIR/voiceforge/" 2>/dev/null || true

log_info "Removing unnecessary files..."
rm -rf "$OUTPUT_DIR/voiceforge/node_modules"
rm -rf "$OUTPUT_DIR/voiceforge/dist"
rm -rf "$OUTPUT_DIR/voiceforge/.git"
rm -rf "$OUTPUT_DIR/voiceforge/.replit"
rm -f "$OUTPUT_DIR/voiceforge/.env"
rm -f "$OUTPUT_DIR/voiceforge/replit.nix"
rm -f "$OUTPUT_DIR/voiceforge/replit.md"
find "$OUTPUT_DIR/voiceforge" -name "*.log" -delete
find "$OUTPUT_DIR/voiceforge" -name ".DS_Store" -delete

log_info "Making scripts executable..."
chmod +x "$OUTPUT_DIR/voiceforge/deploy/"*.sh

log_info "Creating deployment package..."
cd "$OUTPUT_DIR"

if command -v zip &> /dev/null; then
    zip -r "${PACKAGE_NAME}.zip" voiceforge -q
    log_info "Created: ${OUTPUT_DIR}/${PACKAGE_NAME}.zip"
fi

tar -czf "${PACKAGE_NAME}.tar.gz" voiceforge
log_info "Created: ${OUTPUT_DIR}/${PACKAGE_NAME}.tar.gz"

TOTAL_SIZE=$(du -sh "$OUTPUT_DIR/voiceforge" | cut -f1)
log_info "Package size: $TOTAL_SIZE"

echo ""
echo -e "${GREEN}╔═══════════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║          ✓ Package Created Successfully!                  ║${NC}"
echo -e "${GREEN}╠═══════════════════════════════════════════════════════════╣${NC}"
echo -e "${GREEN}║                                                           ║${NC}"
echo -e "${GREEN}║  Output files:                                            ║${NC}"
echo -e "${GREEN}║                                                           ║${NC}"
printf "${GREEN}║  • %-53s ║${NC}\n" "${PACKAGE_NAME}.zip"
printf "${GREEN}║  • %-53s ║${NC}\n" "${PACKAGE_NAME}.tar.gz"
echo -e "${GREEN}║                                                           ║${NC}"
echo -e "${GREEN}║  Location: dist-package/                                  ║${NC}"
echo -e "${GREEN}║                                                           ║${NC}"
echo -e "${GREEN}╠═══════════════════════════════════════════════════════════╣${NC}"
echo -e "${GREEN}║                                                           ║${NC}"
echo -e "${GREEN}║  To deploy on your server:                                ║${NC}"
echo -e "${GREEN}║                                                           ║${NC}"
echo -e "${GREEN}║  1. Upload the zip/tar.gz to your server                  ║${NC}"
echo -e "${GREEN}║  2. Extract: unzip ${PACKAGE_NAME}.zip                 ║${NC}"
echo -e "${GREEN}║  3. Run: cd voiceforge && sudo bash deploy/install.sh     ║${NC}"
echo -e "${GREEN}║                                                           ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════════════════════════╝${NC}"
echo ""
