import { z } from "zod";

export const voiceParametersSchema = z.object({
  pitchMean: z.number(),
  pitchMin: z.number(),
  pitchMax: z.number(),
  pitchStd: z.number(),
  formantF1: z.number(),
  formantF2: z.number(),
  formantF3: z.number(),
  formantF4: z.number(),
  spectralCentroid: z.number(),
  spectralBandwidth: z.number(),
  spectralRolloff: z.number(),
  zeroCrossingRate: z.number(),
  rmsEnergy: z.number(),
  harmonicRatio: z.number(),
  mfccCoefficients: z.array(z.number()),
  tempo: z.number(),
  duration: z.number(),
});

export type VoiceParameters = z.infer<typeof voiceParametersSchema>;

export const transformJobSchema = z.object({
  id: z.string(),
  status: z.enum(["pending", "analyzing_source", "analyzing_target", "extracting_features", "transforming", "matching", "finalizing", "completed", "error"]),
  progress: z.number().min(0).max(100),
  sourceFileName: z.string(),
  targetFileName: z.string(),
  sourceParameters: voiceParametersSchema.optional(),
  targetParameters: voiceParametersSchema.optional(),
  resultParameters: voiceParametersSchema.optional(),
  matchScore: z.number().min(0).max(100).optional(),
  errorMessage: z.string().optional(),
  resultUrl: z.string().optional(),
  intensity: z.number().min(0).max(100).default(100),
});

export type TransformJob = z.infer<typeof transformJobSchema>;

export const uploadResponseSchema = z.object({
  success: z.boolean(),
  fileId: z.string(),
  fileName: z.string(),
  duration: z.number(),
  sampleRate: z.number(),
  channels: z.number(),
  waveformData: z.array(z.number()),
});

export type UploadResponse = z.infer<typeof uploadResponseSchema>;

export const transformRequestSchema = z.object({
  sourceFileId: z.string(),
  targetFileId: z.string(),
  intensity: z.number().min(0).max(100).default(100),
});

export type TransformRequest = z.infer<typeof transformRequestSchema>;

export const batchTransformRequestSchema = z.object({
  sourceFileId: z.string(),
  targetFileIds: z.array(z.string()).min(1).max(10),
  intensity: z.number().min(0).max(100).default(100),
});

export type BatchTransformRequest = z.infer<typeof batchTransformRequestSchema>;

export const batchJobSchema = z.object({
  id: z.string(),
  status: z.enum(["pending", "processing", "completed", "error"]),
  sourceFileName: z.string(),
  totalJobs: z.number(),
  completedJobs: z.number(),
  failedJobs: z.number(),
  jobIds: z.array(z.string()),
  intensity: z.number().min(0).max(100).default(100),
  createdAt: z.number(),
});

export type BatchJob = z.infer<typeof batchJobSchema>;

export const pipelineStages = [
  { id: "analyzing_source", label: "Analyzing Source Voice", icon: "waveform" },
  { id: "analyzing_target", label: "Analyzing Target Voice", icon: "waveform" },
  { id: "extracting_features", label: "Extracting Voice Features", icon: "settings" },
  { id: "transforming", label: "Applying Transformation", icon: "transform" },
  { id: "matching", label: "Iterative Matching", icon: "compare" },
  { id: "finalizing", label: "Finalizing Output", icon: "check" },
] as const;

export type PipelineStage = typeof pipelineStages[number]["id"];
